//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	SIH Inheritances
// Objective: 	Calculate summary statistics for inheritances received in the
//				SIH 2011-12.
//
// Created: 	22/09/2021
// Modified: 	15/11/2021
//==============================================================================

***********
*** SIH ***
***********

clear

cd "XXXX"	// set the content directory here

use "XXXX"	// import the SIH 2011-12 person file data here

gen year = 2012	// generating a year variable

tabstat PERINHE, s(mean median min max count sum)	// tabulating summary statistics about the unweighted inheritances

tabstat PERINHE if PERINHE > 0, s(mean median min max count sum)	// tabulating summary statistics about the unweighted inheritances for positive values of inheritances only

* creating sample-level weights for the SIH
egen pop = sum(SIHPSWT)
egen sample = count(SIHPSWT)

gen POP_INH = PERINHE * SIHPSWT	// generating a population level inheritances variable using the person weights

sum PERINHE [w = SIHPSWT] if PERINHE > 0, detail
gen mean_wt_inh = r(mean) // new variable from the mean from above
gen median_wt_inh = r(p50) // new variabel from the median from above

egen count_wt_inh = sum(SIHPSWT) if PERINHE > 0  // total num of people who inherited
egen sum_wt_inh = sum(POP_INH) if PERINHE > 0 // total amount they inherited

gen wt_percent = count_wt_inh/pop

collapse(mean) year pop sample mean_wt_inh median_wt_inh sum_wt_inh count_wt_inh  wt_percent // collapsing the dataset and retaining the year, mean, median, sum and count variables - note missing will be removed 

save SIH_inheritances.dta, replace	// saving the dataset

*************
*** HILDA ***
*************

clear

use HILDA_restricted_combined_rescaled.dta	// opening the dataset

merge 1:1 xwaveid wave using "bequest_from_HH_death.dta"	// merging in indicator for a bequest that is likely to have come from a household member

keep if (wave == 11 | wave == 12)	// keeping the waves that align with the SIH

xtset xwaveid wave	// declaring the dataset to be a panel

replace oifinha = 0 if oifinha < 0	// replacing negative inheritance values as zero
replace hhwtrps = 0 if hhwtrps < 0	// replacing negative responding-person weights as zero (note: the default weights were used so that the reference populations of HILDA and SIH were more similar)
replace hhwtrp = 0 if hhwtrp < 0	// replacing negative population-level responding-person weiths as zero
// keep if oifinha > 0

sum oifinha [w=hhwtrp] if oifinha>0, detail

gen hilda_mean = r(mean) // new variable from the mean from above
gen hilda_median = r(p50) // new variable from the median from above

gen pop_inherit = oifinha * hhwtrp	// weighting the inheritance variable
egen hilda_sum = sum(pop_inherit)	// calculating the aggregate population inheritances

collapse hilda_mean hilda_median hilda_sum	// collapsing the HILDA dataset to leave just the key variables

gen year = 2012	// generating a year variable for merging

save HILDA_summary_stats_for_merging.dta, replace	// saving the dataset

*******************************************
*** HILDA (per cent receiving bequests) ***
*******************************************

clear

use HILDA_restricted_combined_rescaled.dta	// opening the dataset

merge 1:1 xwaveid wave using "bequest_from_HH_death.dta"	// merging in indicator for a bequest that is likely to have come from a household member

keep if (wave == 11 | wave == 12)	// keeping waves that align with the SIH

xtset xwaveid wave	// declaring the dataset to be a panel

* dropping people who have a missing inheritance observation in either wave
drop if (wave == 11 & F.oifinha == .)
drop if (wave == 12 & L.oifinha == .)

replace oifinha = 0 if oifinha < 0	// replacing negative inheritances as zero
replace hhwtrps = 0 if hhwtrps < 0	// replacing negative weights as zero

gen any_inherit = 1 if wave==12 & (oifinha>0 | L.oifinha>0)

keep xwaveid wave hhwtrps oifinha any_inherit // eyeballing it, looks ok

keep if wave==12 // keep wave 12 only and use wave 12 weights

egen sample = sum(hhwtrps)
egen num_inheritors = sum(hhwtrps) if any_inherit==1

collapse sample num_inheritors

gen hilda_percent = num_inheritors / sample	// generating the per cent of sample members who receive an inheritance across the two waves

gen year = 2012	// generating a year variable for merging

drop sample num_inheritors

save HILDA_percent_for_merging.dta, replace	// saving the dataset

******************************
*** HILDA exc. HH bequests ***
******************************

* The results from this section of the code are not used in Table 1.1 but are referenced in the text surrounding it.

clear

use HILDA_restricted_combined_rescaled.dta	// opening the dataset

merge 1:1 xwaveid wave using "bequest_from_HH_death.dta"	// merging in indicator for a bequest that is likely to have come from a household member

drop if bequest_from_HH_death == 1	// dropping bequests that are likely to have come from a household member and consequently might have been excluded from the SIH

keep if (wave == 11 | wave == 12)	// keeping the waves that align with the SIH

xtset xwaveid wave	// declaring the dataset to be a panel

replace oifinha = 0 if oifinha < 0	// replacing negative inheritance values as zero
replace hhwtrps = 0 if hhwtrps < 0	// replacing negative responding-person weights as zero (note: the default weights were used so that the reference populations of HILDA and SIH were more similar)
replace hhwtrp = 0 if hhwtrp < 0	// replacing negative population-level responding-person weiths as zero
// keep if oifinha > 0

sum oifinha [w=hhwtrp] if oifinha>0, detail

gen hilda_mean_ex_HH = r(mean) // new variable from the mean from above
gen hilda_median_ex_HH = r(p50) // new variable from the median from above

gen pop_inherit_ex_HH = oifinha * hhwtrp	// weighting the inheritance variable
egen hilda_sum_ex_HH = sum(pop_inherit_ex_HH)	// calculating the aggregate population inheritances

collapse hilda_mean_ex_HH hilda_median_ex_HH hilda_sum_ex_HH	// collapsing the HILDA dataset to leave just the key variables

gen year = 2012	// generating a year variable for merging

save HILDA_summary_stats_for_merging_ex_HH.dta, replace	// saving the dataset

************************************************************
*** HILDA exc. HH bequests (per cent receiving bequests) ***
************************************************************

clear

use HILDA_restricted_combined_rescaled.dta	// opening the dataset

merge 1:1 xwaveid wave using "bequest_from_HH_death.dta"	// merging in indicator for a bequest that is likely to have come from a household member

drop if bequest_from_HH_death == 1	// dropping bequests that are likely to have come from a household member and consequently might have been excluded from the SIH

keep if (wave == 11 | wave == 12)	// keeping waves that align with the SIH

xtset xwaveid wave	// declaring the dataset to be a panel

* dropping people who have a missing inheritance observation in either wave
drop if (wave == 11 & F.oifinha == .)
drop if (wave == 12 & L.oifinha == .)

replace oifinha = 0 if oifinha < 0	// replacing negative inheritances as zero
replace hhwtrps = 0 if hhwtrps < 0	// replacing negative weights as zero

gen any_inherit_ex_HH = 1 if wave==12 & (oifinha>0 | L.oifinha>0)

keep xwaveid wave hhwtrps oifinha any_inherit_ex_HH // eyeballing it, looks ok

keep if wave==12 // keep wave 12 only and use wave 12 weights

egen sample_ex_HH = sum(hhwtrps)
egen num_inheritors_ex_HH = sum(hhwtrps) if any_inherit_ex_HH==1

collapse sample_ex_HH num_inheritors_ex_HH

gen hilda_percent_ex_HH = num_inheritors_ex_HH / sample_ex_HH	// generating the per cent of sample members who receive an inheritance across the two waves

gen year = 2012	// generating a year variable for merging

drop sample_ex_HH num_inheritors_ex_HH

save HILDA_percent_for_merging_ex_HH.dta, replace	// saving the dataset

****************************
*** Merging the datasets ***
****************************

merge 1:1 year using HILDA_summary_stats_for_merging_ex_HH.dta, nogen	// merging the datasets

merge 1:1 year using HILDA_percent_for_merging.dta, nogen	// merging the datasets

merge 1:1 year using HILDA_summary_stats_for_merging.dta, nogen	// merging the datasets

merge 1:1 year using SIH_inheritances.dta, nogen	// merging the datasets

drop pop sample count_wt_inh	// dropping non-key variables

* renaming variables
rename mean_wt_inh sih_mean
rename median_wt_inh sih_median
rename sum_wt_inh sih_sum
rename wt_percent sih_percent

order year hilda* sih*	// ordering the dataset

save HILDA_SIH_inheritances_comparison.dta, replace	// saving the dataset